#include <pwd.h>
#include <time.h>
#include <stdlib.h>
#include <string.h>
#include <pthread.h>
#include "list.h"
#include "common.h"
#include "auth.h"

#define AUTHITEM_COMP	0
#define AUTHITEM_SHARE	1

typedef struct{
    char		reserved[sizeof(LIST)];
    int			ref_count;
    hstring		domain;
    hstring		user;
    hstring		password;
    char		data[0];
} AuthData;

typedef struct{
    AuthData	*auth;		// pointer to domain/user/password
    time_t	ctime;		// record change time
} AuthInfo;

typedef struct{
    char	reserved[sizeof(LIST)];
    AuthInfo	info;		
    int		type;		// record type: comp/share
    hstring	name;		// comp/share name
    char	data[0];	// additional data
} authitem;

typedef struct{
    AuthInfo	info;		// default AuthData
    LIST	comp_list;	// list of authitem records for computer
    LIST	auth_data_list;	// list of AuthData records;	
} authroot;

char		login[64]	= "guest";
char		*fake_password	= "********";
authroot	root		= { { NULL, (time_t) 0 }, 
				    STATIC_LIST_INITIALIZER(root.comp_list),
				    STATIC_LIST_INITIALIZER(root.auth_data_list)
				  };
pthread_mutex_t	m_auth		= PTHREAD_MUTEX_INITIALIZER;

void GetUserLogin(){
    register struct passwd	*pw;

    if ((pw = getpwuid(getuid())) == NULL) return;
    safecpy(login, pw->pw_name);
    DPRINT(5, "login=%s\n", login);
}

AuthData* NewAuthData(hstring *domain, hstring *user, hstring *password){
    char	*ptr;
    AuthData	*auth;
    size_t	len;
    
    len = sizeof(AuthData) + domain->length + 
	user->length + password->length + 3;
    if ((auth = malloc(len)) == NULL) return NULL;
    memset(auth, 0, len);
    
    ptr = auth->data;
    strncpy(ptr, domain->string, domain->length);
    memcpy(&auth->domain, domain, sizeof(hstring));
    auth->domain.string = ptr;

    ptr += domain->length + 1;
    strncpy(ptr, user->string, user->length);
    memcpy(&auth->user, user, sizeof(hstring));
    auth->user.string = ptr;

    ptr += user->length + 1;
    strncpy(ptr, password->string, password->length);
    memcpy(&auth->password, password, sizeof(hstring));
    auth->password.string = ptr;

    return auth;
}

AuthData* find_in_auth_list(hstring *domain, hstring *user, hstring *password){
    AuthData	*elem = first_elem(&root.auth_data_list);

    while(is_valid_elem(&root.auth_data_list, elem)){
	if ((hstring_cmp(&elem->domain, domain) == 0) &&
	    (hstring_cmp(&elem->user, user) == 0) &&
	    (hstring_cmp(&elem->password, password) == 0)) return elem;
	elem = next_elem(elem);
    }
    return NULL;
}

AuthData* StoreAuthData(const char *domain, const char *user, const char *password){
    AuthData	*auth;
    hstring	hdomain, huser, hpassword;

    DPRINT(10, "domain=%s, user=%s, password=%s\n", domain, user, fake_password);

    make_hstring(&hdomain,   domain,   strlen(domain));
    make_hstring(&huser,     user,     strlen(user));
    make_hstring(&hpassword, password, strlen(password));

    if ((auth = find_in_auth_list(&hdomain, &huser, &hpassword)) != NULL)
	remove_from_list(&root.auth_data_list, auth)
    else if ((auth = NewAuthData(&hdomain, &huser, &hpassword)) == NULL) return NULL;

    auth->ref_count++;
    add_to_list_back(&root.auth_data_list, auth);
    return auth;
}

void ReleaseAuthData(AuthData *auth){
    DPRINT(10, "domain=%s, user=%s, password=%s\n", 
	auth->domain.string, auth->user.string, fake_password);

    if (auth->ref_count == 0){
	DPRINT(0, "WARNING! trying to release an unused AuthData!\n");
	return;
    }

    auth->ref_count--;
    if (auth->ref_count == 0){
	remove_from_list(&root.auth_data_list, auth);
	free(auth);
    }
}

int UpdateAuthInfo(AuthInfo *info, 
		const char *domain, const char *user, const char *password){

    DPRINT(9, "domain=%s, user=%s, password=%s\n", domain, user, fake_password);

    if (info->auth != NULL) ReleaseAuthData(info->auth);
    info->auth = StoreAuthData(domain, user, password);
    info->ctime = time(NULL);
    return (info->auth == NULL);
}

void ClearAuthInfo(AuthInfo *info){
    if (info->auth != NULL) ReleaseAuthData(info->auth);
    info->auth = NULL;
    info->ctime = time(NULL);
}

authitem* MakeAuthItem(hstring *name, int type){
    char	*ptr;
    authitem	*item;
    size_t	len;

    DPRINT(10, "name=%s, type=%d\n", name->string, type);

    len = sizeof(authitem) + name->length + 1;
    if (type == AUTHITEM_COMP) len += sizeof(LIST);
    if ((item = malloc(len)) == NULL) return NULL;
    memset(item, 0, len);
    item->type = type;

    ptr = item->data;
    if (type == AUTHITEM_COMP) ptr += sizeof(LIST);

    strncpy(ptr, name->string, name->length);
    memcpy(&item->name, name, sizeof(hstring));
    item->name.string = ptr;

    return item;
}

authitem* find_in_authitem_list(LIST *list, hstring *name){
    authitem	*elem = first_elem(list);

    while(is_valid_elem(list, elem)){
	if (hstring_casecmp(&elem->name, name) == 0) return elem;
	elem = next_elem(elem);
    }
    return NULL;
}

// Try to find a computer/share information. If information was not
// found we try to create it. After the search we place requested
// information to the begining of the list to optimise future search
authitem* get_entry(LIST *list, hstring *name, int type){
    authitem	*item;

    if ((item = find_in_authitem_list(list, name)) != NULL){
	if (item->type != type) return NULL;
    	remove_from_list(list, item);
    }else if ((item = MakeAuthItem(name, type)) == NULL) return NULL;

    add_to_list_back(list, item);
    return item;
}

int SetSmbAuthData(const char *server, const char *share,
	    const char *domain, const char *user, const char *password){

    authitem	*item;
    AuthInfo	*info = NULL;
    hstring	hserver, hshare;
    int 	result = 0;

    if (strlen(user) == 0) user = login;

    DPRINT(7, "server=%s, share=%s, domain=%s, user=%s, password=%s\n",
	server, share, domain, user, fake_password);

    make_casehstring(&hserver, server, strlen(server));
    make_casehstring(&hshare,  share,  strlen(share));

    pthread_mutex_lock(&m_auth);
    if (hserver.length > 0){
	item = get_entry(&root.comp_list, &hserver, AUTHITEM_COMP);
	if ((item != NULL) && (hshare.length > 0))
	    item = get_entry((LIST*)item->data, &hshare, AUTHITEM_SHARE);
	if (item != NULL) info = &item->info;
    }else{
	if (hshare.length == 0) info = &root.info;	
    }
    
    if (info != NULL) result = UpdateAuthInfo(info, domain, user, password);
    pthread_mutex_unlock(&m_auth);
    return result;
}

void delete_old_shares(LIST *list, time_t ctime){
    authitem	*tmp, *elem;

    elem = first_elem(list);
    while(is_valid_elem(list, elem)){
	tmp = elem; elem = next_elem(elem);
	if (tmp->info.ctime < ctime){
	    ClearAuthInfo(&tmp->info);
	    remove_from_list(list, tmp);
	    free(tmp);
	}
    }
}

void DeleteOldSmbAuthData(time_t ctime){
    authitem	*tmp, *elem;

    pthread_mutex_lock(&m_auth);
    if (root.info.ctime < ctime) ClearAuthInfo(&root.info);
    elem = first_elem(&root.comp_list);
    while(is_valid_elem(&root.comp_list, elem)){
	tmp = elem; elem = next_elem(elem);
	delete_old_shares((LIST*)tmp->data, ctime);
	if (tmp->info.ctime < ctime){
	    ClearAuthInfo(&tmp->info);
	    if (is_list_empty((LIST*)tmp->data)){
		remove_from_list(&root.comp_list, tmp);
		free(tmp);
	    }
	}
    }
    pthread_mutex_unlock(&m_auth);
}

void smb_auth_fn(
		const char	*server,
		const char	*share,
		char		*wrkgrp, int wrkgrplen,
		char		*user,   int userlen,
		char		*passwd, int passwdlen){

    authitem	*item;
    AuthData	*auth;
    hstring	hserver, hshare;

    make_casehstring(&hserver, server, strlen(server));
    make_casehstring(&hshare,  share,  strlen(share));

    pthread_mutex_lock(&m_auth);
    auth = root.info.auth;
    if ((hserver.length > 0) &&
	((item = find_in_authitem_list(&root.comp_list, &hserver)) != NULL)){

	if (item->info.auth != NULL) auth = item->info.auth;
	if ((hshare.length > 0) &&
	    ((item = find_in_authitem_list((LIST*)item->data, &hshare)) != NULL)){

	    if (item->info.auth != NULL) auth = item->info.auth;
	}
    }
    
    if (auth != NULL){
	DPRINT(5, "server=%s, share=%s, domain=%s, user=%s, password=%s\n",
	    server, share, auth->domain.string, auth->user.string, fake_password);
    
	if (auth->domain.length > 0) safe_copy(wrkgrp, auth->domain.string, wrkgrplen);
	safe_copy(user, auth->user.string, userlen);
	safe_copy(passwd, auth->password.string, passwdlen);
    }else{
	DPRINT(5, "server=%s, share=%s, domain=%s, user=%s, password=%s\n",
	    server, share, "", login, "");

	safe_copy(user, login, userlen);
	safe_copy(passwd, "", passwdlen);
    }
    pthread_mutex_unlock(&m_auth);
}
