#include <errno.h>
#include <wchar.h>
#include <iconv.h>
#include <stdlib.h>
#include <string.h>
#include <pthread.h>
#include <locale.h>
#include <langinfo.h>
#include "common.h"
#include "charset.h"

char	special_char		= '\\';
char	system_charset[64]	= "UTF-8";
char	local_charset[64]	= "UTF-8";
char	samba_charset[64]	= "UTF-8";

iconv_t local2samba		= (iconv_t)(-1);
iconv_t samba2local		= (iconv_t)(-1);

pthread_mutex_t m_charset	= PTHREAD_MUTEX_INITIALIZER;


int set_new_charsets(const char *local, const char *samba){
    iconv_t new_local2samba	= (iconv_t)(-1);
    iconv_t new_samba2local	= (iconv_t)(-1);

    new_local2samba = iconv_open(samba, local);
    if (new_local2samba == (iconv_t)(-1)) goto error0;
    new_samba2local = iconv_open(local, samba);
    if (new_samba2local == (iconv_t)(-1)) goto error1;

    strncpy(local_charset, local, sizeof(local_charset));
    local_charset[sizeof(local_charset) - 1] = '\0';
    strncpy(samba_charset, samba, sizeof(samba_charset));
    samba_charset[sizeof(samba_charset) - 1] = '\0';

    if (local2samba != (iconv_t)(-1)) iconv_close(local2samba);
    if (samba2local != (iconv_t)(-1)) iconv_close(samba2local);

    local2samba = new_local2samba;
    samba2local = new_samba2local;
    return 1;

  error1:
    iconv_close(new_local2samba);
  error0:
    DPRINT(0, "Invalid local or samba charset. Check your config.\n");
    return 0;
}

int charset_init(void){
    static int initialized = 0;

    if (!initialized){
	char	*charset;

	initialized = 1;
	if ((charset = nl_langinfo(CODESET)) != NULL){
	    strncpy(system_charset, charset, sizeof(system_charset));
	    system_charset[sizeof(system_charset) - 1] = '\0';
	    DPRINT(5, "system_charset=%s\n", system_charset);
	}else DPRINT(0, "Can't find system charset, Check your locale.\n");

	strncpy(local_charset, system_charset, sizeof(local_charset));
	local_charset[sizeof(local_charset) - 1] = '\0';
    }

    return ((local2samba == (iconv_t)(-1)) || (samba2local == (iconv_t)(-1))) ?
		set_new_charsets(local_charset, samba_charset) : 1;
};

int set_local_charset(const char *charset){
    int result = 1;

    if ((charset == NULL) || (*charset == '\0')) charset = system_charset;
    pthread_mutex_lock(&m_charset);
    if (strcmp(charset, local_charset) != 0){
	result = set_new_charsets(charset, samba_charset);
	if (result) DPRINT(7, "local_charset=%s\n", charset);
    }
    pthread_mutex_unlock(&m_charset);
    return result;
}

int set_samba_charset(const char *charset){
    int result = 1;

    if ((charset == NULL) || (*charset == '\0')) charset = "UTF-8";
    pthread_mutex_lock(&m_charset);
    if (strcmp(charset, samba_charset) != 0){
	result = set_new_charsets(local_charset, charset);
	if (result) DPRINT(7, "samba_charset=%s\n", charset);
    }
    pthread_mutex_unlock(&m_charset);
    return result;
}

char* strbuf_samba_to_local(iconv_t cd, const char *string, size_t length,
			    char *strbuf, size_t maxlen){
    size_t	i, r, len, out_len;
    char	*in, *out;

    if ((strbuf == NULL) || (maxlen == 0) || (cd == (iconv_t)(-1))) return NULL;

    len = 0;
    while(length > 0){
	in = (char*) string;
	out = strbuf + len;
	out_len = maxlen - len;
	if (out_len == 0) return NULL;

	for(i = 1; i <= length; i++){
	    r = iconv(cd, &in, &i, &out, &out_len);
	    if ((r == 0) && (strbuf[len] != special_char))
		goto conversion_is_ok;
	    if (r == (size_t)(-1)){
		if (errno == E2BIG) return NULL;
		if (errno == EINVAL) continue;
	    }
	    break;
	}

	/* conversion is bad */
	if (maxlen - len < 2) return NULL;
	strbuf[len++] = special_char;
	strbuf[len++] = *string;
	string++; length--;
	continue;

      conversion_is_ok:
	length -= (in - string); string = in;
	len = out - strbuf;
	continue;
    }
    return strbuf;
}

char* strbuf_local_to_samba(iconv_t cd, const char *string, size_t length,
			    char *strbuf, size_t maxlen){
    size_t	i, r, len, out_len;
    char	*in, *out;

    if ((strbuf == NULL) || (maxlen == 0) || (cd == (iconv_t)(-1))) return NULL;

    len = 0;
    while(length > 0){
	in = (char*) string;
	out = strbuf + len;
	out_len = maxlen - len;
	if (out_len == 0) return NULL;

	if (*in == special_char){
	    if (length >= 2) return NULL;
	    string++; length--;
	    strbuf[len++] = *string;
	    string++; length--;
	    continue;
	}

	for(i = 1; i <= length; i++){
	    r = iconv(cd, &in, &i, &out, &out_len);
	    if (r != (size_t)(-1)) goto conversion_is_ok;
	    if (errno == E2BIG) return NULL;
	    if (errno == EINVAL) continue;
	    break;
	}

	/* conversion is bad */
	return NULL;

      conversion_is_ok:
	length -= (in - string); string = in;
	len = out - strbuf;
	continue;
    }
    return strbuf;
}

int local2smb(char *dst, const char *src, size_t dst_len){
    char *res;

    memset(dst, 0, dst_len); dst_len--;
    DPRINT(10,"src=%s\n", src);
    pthread_mutex_lock(&m_charset);
    res = strbuf_local_to_samba(local2samba, src, strlen(src), dst, dst_len);
    if (res == NULL) *dst = '\0';
    pthread_mutex_unlock(&m_charset);
    DPRINT(10,"dst=%s\n", res);
    return (res != NULL);
}

int smb2local(char *dst, const char *src, size_t dst_len){
    char *res;

    memset(dst, 0, dst_len); dst_len--;
    DPRINT(10,"src=%s\n", src);
    pthread_mutex_lock(&m_charset);
    res = strbuf_samba_to_local(samba2local, src, strlen(src), dst, dst_len);
    if (res == NULL) *dst = '\0';
    pthread_mutex_unlock(&m_charset);
    DPRINT(10,"dst=%s\n", res);
    return (res != NULL);
}
