#include "config.h"
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <pthread.h>
#include "common.h"

#ifdef HAVE_BACKTRACE
    #include <execinfo.h>
#endif


int		update_delta		= 5;
int		smbnetfs_debug		= 0;
FILE		*stdlog			= NULL;
pthread_mutex_t m_common		= PTHREAD_MUTEX_INITIALIZER;

int  hash_string(const char *ptr, size_t length){
    int 	hash;
    size_t	i;
    
    for (hash = 0, i = 0; i < length; i++, ptr++)
	hash = (hash << 5) - hash + *ptr;
    return hash;
}

int hstring_cmp(hstring *hs1, hstring *hs2){
    if (hs1->hash != hs2->hash) return 1;
    if (hs1->length != hs2->length) return 1;
    return strncmp(hs1->string, hs2->string, hs1->length);
}

int  hash_stringcase(const char *ptr, size_t length){
    int 	hash;
    size_t	i;
    
    for (hash = 0, i = 0; i < length; i++, ptr++)
	hash = (hash << 5) - hash + toupper(*ptr);
    return hash;
}

int hstring_casecmp(hstring *hs1, hstring *hs2){
    if (hs1->hash != hs2->hash) return 1;
    if (hs1->length != hs2->length) return 1;
    return strncasecmp(hs1->string, hs2->string, hs1->length);
}

void make_hstring(hstring *hs, const char *ptr, size_t length){
    hs->hash = hash_string(ptr, length);
    hs->length = length;
    hs->string = ptr;
}

void make_casehstring(hstring *hs, const char *ptr, size_t length){
    hs->hash = hash_stringcase(ptr, length);
    hs->length = length;
    hs->string = ptr;
}

void safe_copy(char *dst, const char *src, int dst_len){
    if (dst_len == 0) return;
    strncpy(dst, src, dst_len - 1);
    dst[dst_len - 1] = '\0';
};

const char* filename_begin(const char *path){
    while(*path == '/') path++;
    return path;
}

const char* filename_end(const char *path){
    while((*path != '/') && (*path != '\0')) path++;
    return path;
}

size_t filename_len(const char *path){
    path = filename_begin(path);
    return filename_end(path) - path;
}

const char* next_filename(const char *path){
    return filename_begin(filename_end(path));
}


int SetUpdateTimeDelta(int delta){
    DPRINT(7, "delta=%d\n", delta);
    if (delta < 1) return 0;
    pthread_mutex_lock(&m_common);
    update_delta = delta;
    pthread_mutex_unlock(&m_common);
    return 1;
}

int GetUpdateTimeDelta(){
    int delta;
    
    pthread_mutex_lock(&m_common);
    delta = update_delta;
    pthread_mutex_unlock(&m_common);
    DPRINT(7, "delta=%d\n", delta);
    return delta;
}

int SetSMBNetFsDebugLevel(int level){
    DPRINT(7, "level=%d\n", level);
    if ((level < 0) || (level > 10)) return 0;
    pthread_mutex_lock(&m_common);
    smbnetfs_debug = level;
    pthread_mutex_unlock(&m_common);
    return 1;
}

int GetSMBNetFsDebugLevel(){
    int level;
    
    pthread_mutex_lock(&m_common);
    level = smbnetfs_debug;
    pthread_mutex_unlock(&m_common);
    DPRINT(7, "level=%d\n", level);
    return level;
}

int SetLogFile(char *filename){
    DPRINT(7, "filename=%s\n", filename);
    pthread_mutex_lock(&m_common);
    if ((stdlog != NULL) && (stdlog != stderr)) fclose(stdlog);
    if ((filename == NULL) || (*filename == '\0') ||
    	((stdlog = fopen(filename, "a")) == NULL)) stdlog = stderr;
    pthread_mutex_unlock(&m_common);
    return 1;
}

void debug_print(int level, const char *fmt, ...){
    va_list	ap;
    
    pthread_mutex_lock(&m_common);
    if (stdlog == NULL) stdlog = stderr;
    if ((level >= 0) && (level <= smbnetfs_debug)){
	va_start(ap, fmt);
	vfprintf(stdlog, fmt, ap);
	fflush(stdlog);
	va_end(ap);
    }
    pthread_mutex_unlock(&m_common);
}

void print_backtrace(){
#ifdef HAVE_BACKTRACE
    void	*array[200];
    size_t	size;

    DPRINT(0, "dumping backtrace ...\n");

    pthread_mutex_lock(&m_common);
    if (stdlog == NULL) stdlog = stderr;
    size = backtrace (array, 200);
    backtrace_symbols_fd(array, size, fileno(stdlog));
    pthread_mutex_unlock(&m_common);
#endif /* HAVE_BACKTRACE */
}
