#include <config.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <getopt.h>
#include <unistd.h>
#include <string.h>
#include <locale.h>
#include <signal.h>
#include "list.h"
#include "common.h"
#include "buffer.h"
#include "charset.h"
#include "auth.h"
#include "smbctx.h"
#include "smbitem.h"
#include "reconfigure.h"
#include "function.h"

void sig_handler(int signum){
    if (signum == SIGHUP){
	DPRINT(1, "SIGHUP received. Rereading configuration...\n");
	ReadConfig();
	return;
    }
    DPRINT(0, "SIGILL or SIGSEGV received\n");
    print_backtrace();
    exit(signum);
}

void parse_cmdline(int argc, char *argv[]){
    int		c;
    int		print_version = 0;
    
    opterr = 0;
    while(1){
	static struct option	long_options[] = {
	    {"version",		no_argument, NULL, 'V'},
	    {NULL, 0, NULL, 0},
	};
	
	c = getopt_long(argc, argv, "V", long_options, NULL);
	if (c == -1) break;
	
	switch(c){
	    case 'V':
		print_version = 1;
		break;
	    default:
		break;
	}
    }
    
    if (print_version) printf("%s version: %s\n", PACKAGE_NAME, PACKAGE_VERSION);
}

int main(int argc, char *argv[]){

    int 			fd;
    struct sigaction		new_action, old_action;
    struct fuse_file_info	fi;
    
    setlocale(LC_ALL, "");
    parse_cmdline(argc, argv);
    if (!charset_init()){
	fprintf(stderr, "Fatal error: iconv initialization failed.\n");
	exit(1);
    }

    GetUserLogin();
    GetConfigDir();
    
    if (sizeof(fi.fh) < sizeof(char*)){
	fprintf(stderr, "SMBNetFS fatal error:\n"
	    "  There is no space to store pointer in fuse_file_info struct.\n");
	return 1;
    }

    SetUpdateTimeDelta(5);
    SetConfigFileUpdateInterval(300);
    SetSMBNetFsDebugLevel(0);
    SetSmbDebugLevel(0);
    SetSmbItemUpdateInterval(300);
    SetSmbItemTime2Live(900);
    SetMaxSmbCtxCount(15);
    SetSmbCtxUpdateInterval(300);
    SetMaxBufferCount(&OpenFiles, 100);
    SetMaxBufferCount(&ConvertBuffer, 15);
    SetMaxBufferCount(&ArrayData, 1024);
    SetFsBlockSize(32768);
    SetFsQuietFlag(1);
    SetQueryBrowserFlag(1);
    SetShow$Shares(0);
    SetKDEworkaround(3);
    SetLogFile(NULL);
    SetSmbNetbiosName("");
    SetSmbAuthData("", "", "", "", "");
    
    if ((fd = open(".", O_RDONLY)) == -1)
	fprintf(stderr, "Can't remember current directory, error : %s\n"
	    "Current directory will be changed to read config files.\n"
	    "Please use full path to avoid mounting problem next time.\n", 
	    strerror(errno));
    ReadConfig();
    if (fd != -1){
        if (fchdir(fd) == -1)
	    fprintf(stderr, 
		"Current directory was changed to read config files.\n"
		"Please use full path to avoid mounting problem next time.\n");
	close(fd);
    }
    
    AllocateBuffers(&ArrayData);
    AllocateBuffers(&ConvertBuffer);
    AllocateBuffers(&OpenFiles);
    AllocateSmbCtx();

    new_action.sa_handler = sig_handler;
    sigemptyset (&new_action.sa_mask);
    new_action.sa_flags = SA_RESTART;

    if (sigaction (SIGHUP, &new_action, &old_action) < 0){
	fprintf(stderr, "Can't set signal handler\n");
	return 1;
    }
    
    if (sigaction(SIGILL, &new_action, &old_action) < 0){
	fprintf(stderr, "Can't set SIGILL handler\n");
	return 1;
    }

    if (sigaction(SIGSEGV, &new_action, &old_action) < 0){
	fprintf(stderr, "Can't set SIGSEGV handler\n");
	return 1;
    }

    int ret = fuse_main(argc, argv, &smb_oper);
//    int ret = 0;
    
    DeleteOldSmbItem(time(NULL) + 10, USERPATH | AUTOPATH);
    DestroyUnusedCtx();
    DestroyBufferList(&OpenFiles);
    DestroyBufferList(&ConvertBuffer);
    DestroyBufferList(&ArrayData);
    DeleteOldSmbAuthData(time(NULL) + 10);

    return ret;
}
