#include <pwd.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <errno.h>
#include <string.h>
#include <locale.h>
#include <langinfo.h>
#include <sys/stat.h>
#include "common.h"
#include "charset.h"
#include "auth.h"
#include "smbctx.h"
#include "smbitem.h"
#include "function.h"

#define FIELD_MAX	4
#define	PATTERN_SIZE	20
#define LINE_SIZE	200

time_t		last_config_update	= (time_t) 0;
int		config_update_interval	= 300;
char		config_dir[2048]	= "/.smb";
char		*config_file		= "smbnetfs.conf";
pthread_mutex_t	m_config		= PTHREAD_MUTEX_INITIALIZER;

int SetConfigFileUpdateInterval(int interval){
    if (interval < GetUpdateTimeDelta()) return 0;
    pthread_mutex_lock(&m_config);
    config_update_interval = interval;
    pthread_mutex_unlock(&m_config);
    return 1;
}

int GetConfigFileUpdateInterval(){
    int interval;
    
    pthread_mutex_lock(&m_config);
    interval = config_update_interval;
    pthread_mutex_unlock(&m_config);
    return interval;
}

void SetLastConfigUpdateTime(){
    pthread_mutex_lock(&m_config);
    last_config_update = time(NULL);
    pthread_mutex_unlock(&m_config);
}

time_t GetLastConfigUpdateTime(){
    time_t	update_time;
    
    pthread_mutex_lock(&m_config);
    update_time = last_config_update;
    pthread_mutex_unlock(&m_config);
    return update_time;
}

void GetConfigDir(){
    char	*home;
    struct stat buf;

    if ((home = getenv("HOME")) == NULL) return;
    if (strlen(home) + strlen(config_dir) + 1 > sizeof(config_dir)) return;
    memmove(config_dir + strlen(home), config_dir, strlen(config_dir) + 1);
    memcpy(config_dir, home, strlen(home));
    DPRINT(5, "config_dir=%s\n", config_dir);

    if (stat(config_dir, &buf) == 0)
	if (S_ISDIR(buf.st_mode)) return;

    fprintf(stderr, 
	"WARNING!!! Configuration directory ~/.smb is not found. Please create it.\n"
	"This directory should contain at least two files: smb.conf and smbnetfs.conf.\n"
	"You may copy smb.conf from the /etc/samba directory. You can find a sample of\n"
	"smbnetfs.conf in the doc directory of original SMBNetFs distribution.\n\n"
	"Using default settings for now.\n");
}

int get_number(char *value, int *result){
    char *endptr;
    
    *result = strtol(value, &endptr, 0);
    if (*endptr == '\0') return 1;
    else return 0;
}

int set_number(char *value, int (*func)(int)){
    int result;
    
    if (get_number(value, &result)) return func(result);
    else return 0;
}

int set_max_buffer(char *value, BUFFERMANAGER *man){
    int result;
    
    if (get_number(value, &result)) return SetMaxBufferCount(man, result);
    else return 0;
}

int get_boolean(char *value, int *result){
    if ((strcasecmp(value, "true") == 0) || (strcasecmp(value, "yes") == 0)){
	*result = 1;
	return 1;
    }
    if ((strcasecmp(value, "false") == 0) || (strcasecmp(value, "no") == 0)){
	*result = 0;
	return 1;
    }
    return 0;
}

int set_boolean(char *value, int (*func)(int)){
    int result;
    
    if (get_boolean(value, &result)) return func(result);
    else return 0;
}

int split(char *line, char *arg[], int len[], int count){
    int i, l = 0, cnt = 0, mode = '\0', slash = 0;

    for(i = 0; i < count; i++) memset(arg[i], 0, len[i]);
    for( ; *line; line++){
	if (mode == '\0'){
	    if (*line == '#') return cnt;
	    if ((*line == ' ') || (*line == '\t')) continue;
	    cnt++; l = 0; slash = 0; mode = ' ';
	    if ((*line == '"') || (*line == '\'') || (*line == '`')){
		mode = *line;
		continue;
	    }
	}else
	if (mode == ' '){
	    if ((*line == ' ') || (*line == '\t')){
		mode = '\0';
		continue;
	    }
	}else{
	    if (!slash){
		if (*line == '\\'){
		    slash = 1;
		    continue;
		}else
		if (*line == mode){
		    if ((line[1] == ' ') || (line[1] == '\t') ||
					    (line[1] == '\0')){
			mode = '\0';
			continue;
		    }
		    return -cnt;
		}
	    }
	}
        slash = 0; l++; 
	if ((cnt <= count) && (len[cnt - 1] > l)) arg[cnt - 1][l - 1] = *line;
    }
    return cnt;
}

int ReadConfigFile(const char *filename){
    FILE	*file;
    int		ok_permission = 1;
    int		i, cnt, len[FIELD_MAX];
    char	pattern[PATTERN_SIZE], s[LINE_SIZE];
    char	data[FIELD_MAX][LINE_SIZE], *arg[FIELD_MAX];
    const char	*name;
    struct stat	buf;

    if (stat(filename, &buf) != 0){
	DPRINT(3, "Error while reading config %s : %s.\n", 
	    filename, strerror(errno));
	return 0;
    }
    if (!S_ISREG(buf.st_mode)){
	DPRINT(3, "Config %s is not regular file.\n", filename);
	return 0;
    }
    if ((buf.st_uid != getuid()) || ((buf.st_mode & 06177) != 0))
	ok_permission = 0;

    for(i = 0; i < FIELD_MAX; i++){
	arg[i] = data[i];
	len[i] = sizeof(data[i]);
    }
    snprintf(pattern, sizeof(pattern), "%%%d[^\n]\n", (int)sizeof(s) - 1);
    
    if ((file = fopen(filename, "r")) == NULL){
	DPRINT(3, "Open file %s error : %s", filename, strerror(errno));
	return 0;
    }
    fscanf(file, "%[\n]", s);
    while(!feof(file)){
	memset(s, 0, sizeof(s));
	fscanf(file, pattern, s);
	cnt = split(s, arg, len, FIELD_MAX);
	if (cnt == 0) continue;

	if (cnt == 2){
	    if (strcasecmp(arg[0], "include") == 0){
		ReadConfigFile(arg[1]);
		continue;
	    }
	    if (strcasecmp(arg[0], "time_delta") == 0)
		if (set_number(arg[1], SetUpdateTimeDelta)) continue;
	    if (strcasecmp(arg[0], "config_update_interval") == 0)
		if (set_number(arg[1], SetConfigFileUpdateInterval)) continue;
	    if (strcasecmp(arg[0], "smbnetfs_debug") == 0)
		if (set_number(arg[1], SetSMBNetFsDebugLevel)) continue;
	    if (strcasecmp(arg[0], "smb_debug") == 0)
		if (set_number(arg[1], SetSmbDebugLevel)) continue;
	    if (strcasecmp(arg[0], "smb_refresh_time") == 0)
		if (set_number(arg[1], SetSmbItemUpdateInterval)) continue;
	    if (strcasecmp(arg[0], "smb_name_ttl") == 0)
		if (set_number(arg[1], SetSmbItemTime2Live)) continue;
	    if (strcasecmp(arg[0], "smb_ctx_max") == 0)
		if (set_number(arg[1], SetMaxSmbCtxCount)) continue;
	    if (strcasecmp(arg[0], "smb_ctx_refresh_time") == 0)
		if (set_number(arg[1], SetSmbCtxUpdateInterval)) continue;
	    if (strcasecmp(arg[0], "smb_open_files_max") == 0)
		if (set_max_buffer(arg[1], &OpenFiles)) continue;
	    if (strcasecmp(arg[0], "buf_count_max") == 0)
		if (set_max_buffer(arg[1], &ConvertBuffer)) continue;
	    if (strcasecmp(arg[0], "array_slice_max") == 0)
		if (set_max_buffer(arg[1], &ArrayData)) continue;
	    if (strcasecmp(arg[0], "block_size") == 0)
		if (set_number(arg[1], SetFsBlockSize)) continue;
	    if (strcasecmp(arg[0], "quiet_flag") == 0)
		if (set_boolean(arg[1], SetFsQuietFlag)) continue;
	    if (strcasecmp(arg[0], "query_browsers") == 0)
		if (set_boolean(arg[1], SetQueryBrowserFlag)) continue;
	    if (strcasecmp(arg[0], "show_$_shares") == 0)
		if (set_boolean(arg[1], SetShow$Shares)) continue;
	    if (strcasecmp(arg[0], "kde_workaround") == 0)
		if (set_number(arg[1], SetKDEworkaround)) continue;
	    if (strcasecmp(arg[0], "unsafe_truncate") == 0)
		if (set_boolean(arg[1], SetUnsafeTruncate)) continue;
	    if (strcasecmp(arg[0], "local_charset") == 0)
		if (set_local_charset(arg[1])) continue;
	    if (strcasecmp(arg[0], "samba_charset") == 0)
		if (set_samba_charset(arg[1])) continue;
	    if (strcasecmp(arg[0], "log_file") == 0)
		if (SetLogFile(arg[1])) continue;
	    if (strcasecmp(arg[0], "netbios_name") == 0)
		if (SetSmbNetbiosName(arg[1])) continue;
	}
	if ((cnt >= 3) && (cnt <= 4) && (strcasecmp(arg[0], "auth") == 0)){
	    const char		*comp, *share;
	    const char		*workgroup, *user, *password;
	    int			i;

	    if (!ok_permission) goto insecure_permission;
	    if (cnt == 4) {
		comp = filename_begin(arg[1]);
		share = next_filename(comp);
		if (*next_filename(share) != '\0') goto error;
		*((char*)filename_end(comp)) = '\0';
		*((char*)filename_end(share)) = '\0';
		i = 2;
	    }else{
		comp = "";
		share = "";
		i = 1;
	    };

	    if (arg[i][0] == '/') goto error;
	    workgroup = arg[i];
	    if (*filename_end(workgroup) == '/'){
		user = next_filename(workgroup);
		*((char*)filename_end(workgroup)) = '\0';
	    }else{
		workgroup = "";
		user = arg[i];
	    };
	    if (*filename_end(user) != '\0') goto error;

	    password = arg[i + 1];
	    
	    SetSmbAuthData(comp, share, workgroup, user, password);
	    continue;
	}
	if ((cnt == 2) && (strcasecmp(arg[0], "host") == 0)){
	    if (!ok_permission) goto insecure_permission;
	    if (mkhost(arg[1], USERPATH)) continue;
	}
	if ((cnt == 2) && (strcasecmp(arg[0], "group") == 0)){
	    if (!ok_permission) goto insecure_permission;
	    if (mkgroup(arg[1], USERPATH)) continue;
	}
	if ((cnt >= 2) && (cnt <= 3) && (strcasecmp(arg[0], "link") == 0)){
	    if (!ok_permission) goto insecure_permission;
	    if (*arg[2] == '\0'){
		name = filename_begin(arg[1]);
		if (*(name = next_filename(name)) != '\0'){
		    strcpy(arg[2], "../");
		    safe_copy(arg[2] + 3, name, LINE_SIZE - 3);
		}
	    }
	    if (mklink(arg[1], arg[2], USERPATH)) continue;
	}

      error:
	DPRINT(0, "Error: (file: %s) Invalid input line : %s\n", filename, s);
	continue;
	
      insecure_permission:
	DPRINT(0, "Error: Insecure config file permission.\n"
	    "Can't apply '%s' directive.\n"
	    "Run 'chmod 600 %s' to fix it.\n", arg[0], filename);
        continue;
    }
    fclose(file);
    return 1;
}

int ReadConfig(){
    int		result;
    
    SetLastConfigUpdateTime();
    if (chdir(config_dir) != 0) return 0;
    result = ReadConfigFile(config_file);
    DeleteOldSmbItem(GetLastConfigUpdateTime(), USERPATH);
    DeleteOldSmbAuthData(GetLastConfigUpdateTime());
    return result;
}
